/*****************************************************************************
 *
 *  fs_ltt_i.c - Reads .ltt file into internal structure.
 *
 *  Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved.
 *
 *  Confidential information of Monotype Imaging Inc.
 *
 ****************************************************************************/


#include "fs_object.h"
#include "fs_function.h"

#ifdef FS_LINKED_FONTS

#include "fs_ltt.h"
#include "fs_ltt_i.h"
#include "fs_fixed.h"


static FILECHAR *
strnltonewstr(_DS_ FILECHAR *s, size_t n, FILECHAR **e)
{
    FILECHAR *f, *r;
    size_t l;

    f = SYS_MEMCHR(s, '\n', n);
    if (!f)
    {
        *e = s;
        return NULL;
    }
    l = f - s;

    r = FSS_malloc(_PS_ l + 1);
    if (!r)
    {
        *e = s;
        return NULL;
    }
    SYS_STRNCPY(r, s, l);
    r[l] = '\0';

    *e = f;

    return r;
}

static FS_FIXED
strtoFS_FIXED(FILECHAR *s, FILECHAR **e)
{
    FS_FIXED v;
    FS_BOOLEAN isNegative = false;

    v = SYS_STRTOL(s, e, 10);
    if (v < 0)
    {
        /* flag if negative and set positive */
        isNegative = true;
        v *= -1;
    }
    v <<= 16;

    if (**e == '.')
    {
        FS_USHORT i;
        FS_ULONG d;

        s = *e + 1;

        *e = SYS_STRCHR(s, '\n');

        if (*e == 0)
            *e = s; /* should not happen */

        d = 1;

        for (i = 0; (s < *e) && (i < 7); i++, s++)
        {
            d *= 10;
            v += (((*s - '0') << 16) + (d >> 1)) / d;
        }
    }

    /* if the string was negative, multiply by -1 FIXED (0xFFFF0000) */
    if (isNegative)
        v = FixMul(v, 0xFFFF0000); /*lint !e569*/

    return v;
}

static FS_ERROR
FsLtt_versionFromBuffer(FILECHAR *buffer, FS_USHORT *version)
{
    if (SYS_STRNCMP("LTT\n", buffer, 4) != 0)
    {
        return ERR_NO_MATCHING;
    }
    *version = (FS_USHORT)SYS_STRTOUL(buffer + 4, NULL, 0);

    if (*version > CURRENT_LTT_VERSION)
    {
        return ERR_BAD_FONT_TYPE;
    }
    return SUCCESS;
}

static FS_ERROR
FsLtt_headFromBuffer(FsLtt *ltt, FILECHAR *buffer, size_t headSize)
{
    FILECHAR *s, *e;
    size_t r;
    FS_USHORT i;
    LTT_STATE_PTR;

    if (SYS_STRNCMP("LTT\n", buffer, 4) != 0)
    {
        return ERR_NO_MATCHING;
    }
    s = buffer + 4;

    ltt->version = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

    s = e + 1;

    if (ltt->version == 1)
    {
        ltt->numLegal = 1;
    }
    else
    {
        ltt->numLegal = (FS_USHORT)SYS_STRTOUL(s, &e, 0);
        if (!((ltt->numLegal > 0) && (ltt->numLegal < 10))) /* Klocwork fix */
        {
            return ERR_BAD_LTT_INDEX;
        }
        s = e + 1;
    }
#ifdef FS_MEM_DBG
    STATE.memdbgid = "ltt->legal";
#endif
    ltt->legal = FSS_malloc(_PS_ ltt->numLegal * sizeof(FILECHAR *));
    if (!ltt->legal)    /* Klocwork fix */
    {
        return ERR_MALLOC_FAIL;
    }
    for (i = 0; i < ltt->numLegal; i++)
    {
#ifdef FS_MEM_DBG
        STATE.memdbgid = "ltt->legal[i]";
#endif
        ltt->legal[i] = strnltonewstr(_PS_ s, headSize - (s - buffer), &e);

        s = e + 1;
    }
#ifdef FS_MEM_DBG
    STATE.memdbgid = "ltt->name";
#endif
    ltt->name = strnltonewstr(_PS_ s, headSize - (s - buffer), &e);

    s = e + 1;

    ltt->numComponents = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

    s = e + 1;

    ltt->mtrxComponent = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

    s = e + 1;

    for (r = headSize - (s - buffer); r > 0; r = headSize - (s - buffer))
    {
        if (SYS_STRNCMP("subfamily=", s, 10) == 0)
        {
            if (ltt->subfamilyName) FSS_free(_PS_ ltt->subfamilyName); /* Klocwork fix */

            s += 10;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "ltt->subfamilyName";
#endif
            ltt->subfamilyName = strnltonewstr(_PS_ s, headSize - (s - buffer), &e);

            s = e + 1;
        }
        else if (SYS_STRNCMP("fileversion=", s, 12) == 0)
        {
            if (ltt->fileversion) FSS_free(_PS_ ltt->fileversion); /* Klocwork fix */

            s += 12;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "ltt->fileversion";
#endif
            ltt->fileversion = strnltonewstr(_PS_ s, headSize - (s - buffer), &e);

            s = e + 1;
        }
        else if (SYS_STRNCMP("name_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->ttf_name.offset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("name_size=", s, 10) == 0)
        {
            s += 10;

            ltt->ttf_name.size = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("maxp_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->maxp_off = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("os/2_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->os_2_off = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("head_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->head_off = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("hhea_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->hhea_off = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("vhea_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->vhea_off = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("post_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->post_off = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("cmap_offset=", s, 12) == 0)
        {
            s += 12;

            ltt->cmap.offset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("cmap_size=", s, 10) == 0)
        {
            s += 10;

            ltt->cmap.size = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("ot_gdef_offset=", s, 15) == 0)
        {
            s += 15;

            ltt->gdef.offset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("ot_gdef_size=", s, 13) == 0)
        {
            s += 13;

            ltt->gdef.size = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("ot_gsub_offset=", s, 15) == 0)
        {
            s += 15;

            ltt->gsub.offset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("ot_gsub_size=", s, 13) == 0)
        {
            s += 13;

            ltt->gsub.size = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("ot_gpos_offset=", s, 15) == 0)
        {
            s += 15;

            ltt->gpos.offset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("ot_gpos_size=", s, 13) == 0)
        {
            s += 13;

            ltt->gpos.size = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("grouptypesoffset=", s, 17) == 0)
        {
            s += 17;

            ltt->groupTypesOffset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;

        }
        else if (SYS_STRNCMP("grouptypescount=", s, 16) == 0)
        {
            s += 16;

            ltt->numGroupTypes = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else
        {
            e = SYS_MEMCHR(s, '\n', r);
            if (!e)
            {
                s = buffer + headSize;
            }
            else
            {
                s = e + 1;
            }
        }
    }
    if (ltt->version <= PRE_FOUR_ONE_LTT_VERSION)
    {
        ltt->maxp_off = FSLTT_PRE7_HEAD_SIZE + (ltt->numComponents * FSLTT_COMP_DESC_SIZE);
    }
    if (ltt->numComponents < 1)
    {
        ltt->components = NULL;
        ltt->capacityComponents = 0;
        return ERR_BAD_LTT_NUM_FONTS;
    }
#ifdef FS_MEM_DBG
    STATE.memdbgid = "ltt->components";
#endif
    ltt->components = (FsLttComponent *)FSS_malloc(_PS_
                      ltt->numComponents * sizeof(FsLttComponent));
    if (!ltt->components)
    {
        return ERR_MALLOC_FAIL;
    }
    ltt->capacityComponents = ltt->numComponents;

    return SUCCESS;
}

static FS_ERROR
FsLttComponent_fromBuffer(_DS_ FS_USHORT version,
                          FsLttComponent *lttc, FILECHAR *buffer)
{
    FILECHAR *s, *e;
    size_t r;

    s = buffer;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "lttc->fontName";
#endif
    lttc->fontName = strnltonewstr(_PS_ s, FSLTT_COMP_DESC_SIZE - (s - buffer), &e);

    s = e + 1;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "lttc->fileName";
#endif
    lttc->fileName = strnltonewstr(_PS_ s, FSLTT_COMP_DESC_SIZE - (s - buffer), &e);

    s = e + 1;

    lttc->bmp.range1 = SYS_STRTOUL(s, &e, 0);

    s = e + 1;

    lttc->bmp.range2 = SYS_STRTOUL(s, &e, 0);

    s = e + 1;

    lttc->numGlyphs = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

    s = e + 1;

    if (version <= PRE_NUMICONS_LTT_VERSION)
    {
        lttc->numIcons = 0;
    }
    else
    {
        lttc->numIcons = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

        s = e + 1;
    }
    for (r = FSLTT_COMP_DESC_SIZE - (s - buffer); r > 0; r = FSLTT_COMP_DESC_SIZE - (s - buffer))
    {
        if (SYS_STRNCMP("ttcindex=", s, 9) == 0)
        {
            s += 9;

            lttc->ttcIndex = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("targetcallbackID=", s, 17) == 0)
        {
            if (lttc->targetCId) FSS_free(_PS_ lttc->targetCId); /* Klocwork fix */

            s += 17;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "lttc->targetCId";
#endif
            lttc->targetCId = strnltonewstr(_PS_ s, r - 17, &e);

            s = e + 1;
        }
        else if (SYS_STRNCMP("targetdirectory=", s, 16) == 0)
        {
            if (lttc->targetDir) FSS_free(_PS_ lttc->targetDir); /* Klocwork fix */

            s += 16;

#ifdef FS_MEM_DBG
            STATE.memdbgid = "lttc->targetDir";
#endif
            lttc->targetDir = strnltonewstr(_PS_ s, r - 16, &e);

            s = e + 1;
        }
        else if (SYS_STRNCMP("targetmemptr=", s, 13) == 0)
        {
            s += 13;

            lttc->targetPtr = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("targetoffset=", s, 13) == 0)
        {
            s += 13;

            lttc->targetOff = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("targetlength=", s, 13) == 0)
        {
            s += 13;

            lttc->targetLen = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("embeddedoffset=", s, 15) == 0)
        {
            s += 15;

            lttc->fontOffset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("embeddedsize=", s, 13) == 0)
        {
            s += 13;

            lttc->fontSize = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("bold=", s, 5) == 0)
        {
            s += 5;

            lttc->bold = (FS_BOOLEAN)SYS_STRTOL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("italic=", s, 7) == 0)
        {
            s += 7;

            lttc->italic = (FS_BOOLEAN)SYS_STRTOL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("boldpercent=", s, 12) == 0)
        {
            s += 12;

            lttc->boldPercent = strtoFS_FIXED(s, &e);

            s = e + 1;
        }
        else if (SYS_STRNCMP("italicangle=", s, 12) == 0)
        {
            s += 12;

            lttc->italicAngle = strtoFS_FIXED(s, &e);

            s = e + 1;
        }
#ifdef STROKE_PERCENT_PER_COMPONENT_IS_DESIRED
        else if (SYS_STRNCMP("strokepercent=", s, 14) == 0)
        {
            s += 14;

            lttc->strokePercent = strtoFS_FIXED(s, &e);

            s = e + 1;
        }
#endif
        else if (SYS_STRNCMP("groupprimary=", s, 13) == 0)
        {
            s += 13;

            lttc->group = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("plane1range1=", s, 13) == 0)
        {
            s += 13;

            lttc->smp.range1 = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("plane1range2=", s, 13) == 0)
        {
            s += 13;

            lttc->smp.range2 = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("plane2range1=", s, 13) == 0)
        {
            s += 13;

            lttc->sip.range1 = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("plane2range2=", s, 13) == 0)
        {
            s += 13;

            lttc->sip.range2 = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("plane14range1=", s, 14) == 0)
        {
            s += 14;

            lttc->ssp.range1 = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("plane14range2=", s, 14) == 0)
        {
            s += 14;

            lttc->ssp.range2 = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("adjustmentsoffset=", s, 18) == 0)
        {
            s += 18;

            lttc->adjustmentsOffset = SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else if (SYS_STRNCMP("adjustmentscount=", s, 17) == 0)
        {
            s += 17;

            lttc->numAdjustments = (FS_USHORT)SYS_STRTOUL(s, &e, 0);

            s = e + 1;
        }
        else
        {
            e = SYS_MEMCHR(s, '\n', r);
            if (!e)
            {
                s = buffer + FSLTT_COMP_DESC_SIZE;
            }
            else
            {
                s = e + 1;
            }
        }
    }
    return SUCCESS;
}

/* These are unused
#define MY_COPYL(b) (((b)[0] << 24) | ((b)[1] << 16) | ((b)[2] <<  8) |  (b)[3]); (b) += 4
#define MY_COPYW(b) (((b)[0] <<  8) |  (b)[1]); (b) += 2
*/

static FS_ERROR
FsLtt_readHead(_DS_ FsLtt *ltt, FS_FILE *fp)
{
    FILECHAR *buffer;
    size_t n;
    FS_ERROR e;
    FS_USHORT version;
    size_t headSize;

    /*
     *  this used to contain the "buffer" on the stack, avoiding a costly
     *  malloc/free situation, but 512 bytes on the stack is considered
     *  too big for some environments.
     */

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FSLTT_FILE_HEAD_SIZE";
#endif
    /* allocate "+ 1" for NULL to eliminate Parasoft Insure++ complaint */
    buffer = (FILECHAR *)FSS_malloc(_PS_ FSLTT_FILE_HEAD_SIZE + 1);
    if (!buffer)
    {
        return STATE.error;
    }
    n = FS_read(_PS_ fp, (FS_BYTE *)buffer, FSLTT_PRE7_HEAD_SIZE);
    if (n != FSLTT_PRE7_HEAD_SIZE)
    {
        FSS_free(_PS_ buffer);
        return ERR_FILE_READ;
    }
    buffer[FSLTT_PRE7_HEAD_SIZE] = 0; /* fix for Parasoft Insure++ */

    e = FsLtt_versionFromBuffer(buffer, &version);
    if (e != SUCCESS)
    {
        FSS_free(_PS_ buffer);
        return e;
    }
    if (version > PRE_HEAD_SIZE_INC_VERSION)
    {
        n = FS_read(_PS_ fp,
                    (FS_BYTE *)buffer + FSLTT_PRE7_HEAD_SIZE,
                    FSLTT_FILE_HEAD_SIZE - FSLTT_PRE7_HEAD_SIZE);
        if (n != (FSLTT_FILE_HEAD_SIZE - FSLTT_PRE7_HEAD_SIZE))
        {
            FSS_free(_PS_ buffer);
            return ERR_FILE_READ;
        }
        buffer[FSLTT_FILE_HEAD_SIZE] = 0; /* fix for Parasoft Insure++ */

        headSize = FSLTT_FILE_HEAD_SIZE;
    }
    else headSize = FSLTT_PRE7_HEAD_SIZE;

    e = FsLtt_headFromBuffer(ltt, buffer, headSize);
    if (e != SUCCESS)
    {
        FSS_free(_PS_ buffer);
        return e;
    }
    FSS_free(_PS_ buffer);

    return SUCCESS;
}

static FS_ERROR
FsLttComponent_read(_DS_ FS_USHORT version, FsLttComponent *lttc, FS_FILE *fp)
{
    FILECHAR *buffer;
    size_t n;
    FS_ERROR e;

    /*
     *  this used to contain the "buffer" on the stack, avoiding a costly
     *  malloc/free situation, but 512 bytes on the stack is considered
     *  too big for some environments.
     */

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FSLTT_COMP_DESC_SIZE";
#endif
    /* allocate "+ 1" for NULL to eliminate Parasoft Insure++ complaint */
    buffer = (FILECHAR *)FSS_malloc(_PS_ FSLTT_COMP_DESC_SIZE + 1);
    if (!buffer)
        return STATE.error;

    n = FS_read(_PS_ fp, (FS_BYTE *)buffer, FSLTT_COMP_DESC_SIZE);
    if (n != FSLTT_COMP_DESC_SIZE)
    {
        FSS_free(_PS_ buffer);
        return ERR_FILE_READ;
    }
    buffer[FSLTT_COMP_DESC_SIZE] = 0; /* fix for Parasoft Insure++ */

    e = FsLttComponent_fromBuffer(_PS_ version, lttc, buffer);
    FSS_free(_PS_ buffer);
    if (e != SUCCESS)
    {
        return e;
    }
    return SUCCESS;
}

static FS_ERROR
FsPPemAdjust_fromBuffer(FsPPemAdjust *a, FILECHAR *b)
{
    a->ppem  = (b[0] << 8) | b[1];
    b += 2;
    a->scale = (b[0] << 8) | b[1];
    b += 2;
    a->shift = (b[0] << 8) | b[1];

    return SUCCESS;
}

static FS_ERROR
FsGroupType_fromBuffer(FsGroupType *a, FILECHAR *b)
{
    a->group = (b[0] << 8) | b[1];
    b += 2;
    a->type  = (b[0] << 8) | b[1];

    return SUCCESS;
}

FS_ERROR
FsLtt_read(FsLtt *ltt, FS_FILE *fp, FS_BOOLEAN loadAdjustments)
{
    FS_ERROR e;
    FS_USHORT i;
    LTT_STATE_PTR;

    e = FsLtt_readHead(_PS_ ltt, fp);
    if (e != SUCCESS)
    {
        return e;
    }
    for (i = 0; i < ltt->numComponents; i++)
    {
        FsLttComponent_init(ltt->components + i, ltt);

        e = FsLttComponent_read(_PS_ ltt->version, ltt->components + i, fp);
        if (e != SUCCESS)
        {
            return e;
        }
    }
    if (loadAdjustments)
    {
        for (i = 0; i < ltt->numComponents; i++)
        {
            FsLttComponent *lttc = ltt->components + i;

            if (lttc->numAdjustments)
            {
                FS_USHORT j;

                lttc->adjustments = FSS_malloc(_PS_ lttc->numAdjustments * sizeof(FsPPemAdjust));
                if (!lttc->adjustments)
                {
                    return e;
                }
                lttc->adjustmentsCapacity = lttc->numAdjustments;

                FS_seek(_PS_ fp, lttc->adjustmentsOffset, SEEK_SET);

                for (j = 0; j < lttc->numAdjustments; j++)
                {
                    size_t n;
                    FILECHAR buffer[sizeof(FsPPemAdjust)];
                    FsPPemAdjust *a = lttc->adjustments + j;

                    n = FS_read(_PS_ fp, (FS_BYTE *)buffer, sizeof(FsPPemAdjust));
                    if (n != sizeof(FsPPemAdjust))
                    {
                        return ERR_FILE_READ;
                    }
                    e = FsPPemAdjust_fromBuffer(a, buffer);
                    if (e != SUCCESS)
                    {
                        return e;
                    }
                }
            }
        }
    }
    if (ltt->numGroupTypes)
    {
        FS_USHORT j;

        ltt->groupTypes = FSS_malloc(_PS_ ltt->numGroupTypes * sizeof(FsGroupType));
        if (!ltt->groupTypes)
        {
            return e;
        }
        ltt->groupTypesCapacity = ltt->numGroupTypes;

        FS_seek(_PS_ fp, ltt->groupTypesOffset, SEEK_SET);

        for (j = 0; j < ltt->numGroupTypes; j++)
        {
            size_t n;
            FILECHAR buffer[sizeof(FsGroupType)];
            FsGroupType *a = ltt->groupTypes + j;

            n = FS_read(_PS_ fp, (FS_BYTE *)buffer, sizeof(FsGroupType));
            if (n != sizeof(FsGroupType))
            {
                return ERR_FILE_READ;
            }
            e = FsGroupType_fromBuffer(a, buffer);
            if (e != SUCCESS)
            {
                return e;
            }
        }
    }
    return SUCCESS;
}

FS_ERROR
FsLtt_load(FsLtt *ltt, FS_BYTE *mem, FS_BOOLEAN loadAdjustments)
{
    FS_ERROR e;
    FS_USHORT i;
    FS_USHORT version;
    size_t headSize;
    FS_BYTE *begin = mem;
    LTT_STATE_PTR;

    e = FsLtt_versionFromBuffer((FILECHAR *)mem, &version);
    if (e != SUCCESS)
    {
        return e;
    }
    if (version > PRE_HEAD_SIZE_INC_VERSION)
    {
        headSize = FSLTT_FILE_HEAD_SIZE;
    }
    else headSize = FSLTT_PRE7_HEAD_SIZE;

    e = FsLtt_headFromBuffer(ltt, (FILECHAR *)mem, headSize);
    if (e != SUCCESS)
    {
        return e;
    }
    mem += headSize;

    for (i = 0; i < ltt->numComponents; i++)
    {
        FsLttComponent_init(ltt->components + i, ltt);

        e = FsLttComponent_fromBuffer(_PS_ ltt->version,
                                      ltt->components + i, (FILECHAR *)mem);
        if (e != SUCCESS)
        {
            return e;
        }
        mem += FSLTT_COMP_DESC_SIZE;
    }
    if (loadAdjustments)
    {
        for (i = 0; i < ltt->numComponents; i++)
        {
            FsLttComponent *lttc = ltt->components + i;

            if (lttc->numAdjustments)
            {
                FS_USHORT j;

                lttc->adjustments = FSS_malloc(_PS_ lttc->numAdjustments * sizeof(FsPPemAdjust));
                if (!lttc->adjustments)
                {
                    return e;
                }
                lttc->adjustmentsCapacity = lttc->numAdjustments;

                mem = begin + lttc->adjustmentsOffset;

                for (j = 0; j < lttc->numAdjustments; j++)
                {
                    FsPPemAdjust *a = lttc->adjustments + j;

                    e = FsPPemAdjust_fromBuffer(a, (FILECHAR *)mem);
                    if (e != SUCCESS)
                    {
                        return e;
                    }
                    mem += sizeof(FsPPemAdjust);
                }
            }
        }
    }
    if (ltt->numGroupTypes)
    {
        FS_USHORT j;

        ltt->groupTypes = FSS_malloc(_PS_ ltt->numGroupTypes * sizeof(FsGroupType));
        if (!ltt->groupTypes)
        {
            return e;
        }
        ltt->groupTypesCapacity = ltt->numGroupTypes;

        if (ltt->groupTypesOffset < 0xFFFF0000)
            mem = begin + ltt->groupTypesOffset;
        else
            return ERR_INVALID_FILE_SIZE;

        for (j = 0; j < ltt->numGroupTypes; j++)
        {
            FsGroupType *a = ltt->groupTypes + j;

            e = FsGroupType_fromBuffer(a, (FILECHAR *)mem);
            if (e != SUCCESS)
            {
                return e;
            }
            mem += sizeof(FsGroupType);
        }
    }
    return SUCCESS;
}

#endif /* FS_LINKED_FONTS */
